<?php
/**
 * Plugin Name: CODARAB - Block Payment Gateways by Country (IP FIRST - Works on Blocks)
 * Description: Blocks payment gateways by visitor IP country (primary). Works on Classic + Checkout Block (Store API). Includes GeoIP fallback via ipapi.co.
 * Version: 3.0.0
 * Author: CODARAB DEV
 */

if (!defined('ABSPATH')) exit;

class CODARAB_Block_Gateways_By_Country {
  const OPT = 'codarab_bgc_rules';
  const NONCE_ACTION = 'codarab_bgc_save';

  public function __construct() {
    add_action('admin_menu', [$this, 'admin_menu']);
    add_action('admin_init', [$this, 'handle_save']);

    // Classic checkout: hide gateways
    add_filter('woocommerce_available_payment_gateways', [$this, 'filter_gateways_classic'], 99999);

    // Classic checkout: hard block (Place order)
    add_action('woocommerce_checkout_process', [$this, 'classic_hard_block'], 99999);

    // ✅ Blocks/Store API: hide gateways in Checkout Block
    add_filter('woocommerce_store_api_payment_gateways', [$this, 'store_api_filter_gateways'], 99999);

    // ✅ Blocks/Store API: hard block checkout (Place order)
    add_filter('woocommerce_store_api_checkout_update_order_from_request', [$this, 'store_api_hard_block'], 99999, 2);

    // No-cache on checkout/store-api
    add_action('wp', [$this, 'set_nocache_on_sensitive_pages'], 0);
  }

  /* ---------------- Rules ---------------- */

  private function get_rules() {
    $rules = get_option(self::OPT, []);
    if (!is_array($rules)) $rules = [];

    $clean = [];
    foreach ($rules as $r) {
      $c = isset($r['country']) ? strtoupper(sanitize_text_field($r['country'])) : '';
      $g = isset($r['gateway']) ? sanitize_text_field($r['gateway']) : '';
      if ($c && $g) $clean[] = ['country' => $c, 'gateway' => $g];
    }
    return $clean;
  }

  private function blocked_gateway_ids_for_country($country) {
    $country = strtoupper($country);
    $blocked = [];
    foreach ($this->get_rules() as $r) {
      if ($r['country'] === $country) $blocked[] = $r['gateway'];
    }
    return array_values(array_unique($blocked));
  }

  private function is_gateway_blocked_for_country($country, $gateway_id) {
    $country = strtoupper($country);
    $gateway_id = sanitize_text_field($gateway_id);
    foreach ($this->get_rules() as $r) {
      if ($r['country'] === $country && $r['gateway'] === $gateway_id) return true;
    }
    return false;
  }

  /* ---------------- IP + Country (IP FIRST) ---------------- */

  private function get_ip() {
    if (class_exists('WC_Geolocation')) {
      $ip = WC_Geolocation::get_ip_address();
      if (!empty($ip)) return $ip;
    }
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
      $parts = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
      foreach ($parts as $p) {
        $ip = trim($p);
        if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
      }
    }
    return !empty($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : '';
  }

  private function ip_country() {
    // Cache per IP for 24h
    $ip = $this->get_ip();
    if (!$ip) return '';

    $key = 'codarab_bgc_ip_country_' . md5($ip);
    $cached = get_transient($key);
    if (!empty($cached)) return strtoupper($cached);

    // 1) Try WooCommerce GeoIP first
    if (function_exists('WC') && WC()->geolocation) {
      $geo = WC()->geolocation->geolocate_ip($ip);
      if (!empty($geo['country'])) {
        $cc = strtoupper($geo['country']);
        set_transient($key, $cc, 24 * HOUR_IN_SECONDS);
        return $cc;
      }
    }

    // 2) Fallback to ipapi.co
    $resp = wp_remote_get('https://ipapi.co/' . rawurlencode($ip) . '/country/', [
      'timeout' => 5,
      'headers' => ['User-Agent' => 'WordPress/CODARAB-BGC']
    ]);
    if (!is_wp_error($resp)) {
      $cc = strtoupper(trim(wp_remote_retrieve_body($resp)));
      if (preg_match('/^[A-Z]{2}$/', $cc)) {
        set_transient($key, $cc, 24 * HOUR_IN_SECONDS);
        return $cc;
      }
    }

    return '';
  }

  /* ---------------- No-cache ---------------- */

  public function set_nocache_on_sensitive_pages() {
    $uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
    $is_store_api = (strpos($uri, '/wc/store/') !== false);

    if ((function_exists('is_checkout') && is_checkout()) || $is_store_api) {
      if (!defined('DONOTCACHEPAGE')) define('DONOTCACHEPAGE', true);
      if (function_exists('nocache_headers')) nocache_headers();
      @header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
      @header('Pragma: no-cache');
      @header('X-LiteSpeed-Cache-Control: no-cache');
    }
  }

  /* ---------------- Classic checkout ---------------- */

  public function filter_gateways_classic($available_gateways) {
    if (!is_array($available_gateways)) return $available_gateways;
    if (is_admin() && !wp_doing_ajax()) return $available_gateways;

    $country = $this->ip_country(); // ✅ IP FIRST
    if (!$country) return $available_gateways;

    $blocked = $this->blocked_gateway_ids_for_country($country);
    if (!$blocked) return $available_gateways;

    foreach ($blocked as $gateway_id) {
      if (isset($available_gateways[$gateway_id])) unset($available_gateways[$gateway_id]);
    }
    return $available_gateways;
  }

  public function classic_hard_block() {
    $country = $this->ip_country(); // ✅ IP FIRST
    if (!$country) return;

    $chosen = '';
    if (!empty($_POST['payment_method'])) $chosen = sanitize_text_field($_POST['payment_method']);
    if (!$chosen && function_exists('WC') && WC()->session) {
      $sess = WC()->session->get('chosen_payment_method');
      if (!empty($sess)) $chosen = sanitize_text_field($sess);
    }

    if ($chosen && $this->is_gateway_blocked_for_country($country, $chosen)) {
      wc_add_notice(__('This payment method is not available in your country.', 'codarab-bgc'), 'error');
    }
  }

  /* ---------------- Blocks / Store API ---------------- */

  public function store_api_filter_gateways($gateways) {
    if (!is_array($gateways)) return $gateways;

    $country = $this->ip_country(); // ✅ IP FIRST
    if (!$country) return $gateways;

    $blocked = $this->blocked_gateway_ids_for_country($country);
    if (!$blocked) return $gateways;

    foreach ($blocked as $gateway_id) {
      if (isset($gateways[$gateway_id])) unset($gateways[$gateway_id]);
    }
    return $gateways;
  }

  public function store_api_hard_block($order, $request) {
    $country = $this->ip_country(); // ✅ IP FIRST
    if (!$country) return $order;

    $chosen = '';
    if (is_object($request) && method_exists($request, 'get_param')) {
      $chosen = $request->get_param('payment_method');
      if (!$chosen) $chosen = $request->get_param('paymentMethod');

      // Some blocks send payment_data array
      if (!$chosen) {
        $payment_data = $request->get_param('payment_data');
        if (is_array($payment_data)) {
          foreach ($payment_data as $row) {
            if (is_array($row) && isset($row['key'], $row['value']) && $row['key'] === 'payment_method') {
              $chosen = $row['value'];
              break;
            }
          }
        }
      }
    }

    $chosen = $chosen ? sanitize_text_field($chosen) : '';
    if ($chosen && $this->is_gateway_blocked_for_country($country, $chosen)) {
      throw new Exception(__('This payment method is not available in your country.', 'codarab-bgc'));
    }
    return $order;
  }

  /* ---------------- Admin ---------------- */

  public function admin_menu() {
    add_submenu_page(
      'woocommerce',
      'Block Gateways by Country',
      'Block Gateways by Country',
      'manage_woocommerce',
      'codarab-block-gateways-country',
      [$this, 'admin_page']
    );
  }

  public function admin_page() {
    if (!current_user_can('manage_woocommerce')) return;

    $rules = $this->get_rules();
    $countries = function_exists('WC') ? WC()->countries->get_countries() : [];
    $gateways  = (function_exists('WC') && WC()->payment_gateways())
      ? WC()->payment_gateways()->payment_gateways()
      : [];

    $debug_ip = $this->get_ip();
    $debug_cc = $this->ip_country();

    ?>
    <div class="wrap">
      <h1>Block Payment Gateways by Country (IP FIRST)</h1>

      <?php if (!empty($_GET['saved'])): ?>
        <div class="notice notice-success is-dismissible"><p>Rules saved.</p></div>
      <?php endif; ?>

      <div style="background:#fff;border:1px solid #ccd0d4;padding:12px 14px;border-radius:8px;max-width:980px;">
        <h2 style="margin:0 0 8px;">Debug</h2>
        <p style="margin:0;">
          <strong>Detected IP:</strong> <code><?php echo esc_html($debug_ip ?: 'EMPTY'); ?></code><br>
          <strong>IP Country Used:</strong> <code><?php echo esc_html($debug_cc ?: 'EMPTY'); ?></code><br>
          <small>This plugin blocks by IP country.</small>
        </p>
      </div>

      <hr>

      <h2>Add a rule</h2>
      <form method="post">
        <?php wp_nonce_field(self::NONCE_ACTION, 'codarab_bgc_nonce'); ?>
        <input type="hidden" name="codarab_bgc_action" value="add_rule">

        <table class="form-table">
          <tr>
            <th><label for="country">Country</label></th>
            <td>
              <select name="country" id="country" required>
                <option value="">— Select country —</option>
                <?php foreach ($countries as $code => $name): ?>
                  <option value="<?php echo esc_attr($code); ?>"><?php echo esc_html($name . " ($code)"); ?></option>
                <?php endforeach; ?>
              </select>
            </td>
          </tr>
          <tr>
            <th><label for="gateway">Gateway</label></th>
            <td>
              <select name="gateway" id="gateway" required>
                <option value="">— Select gateway —</option>
                <?php foreach ($gateways as $gw_id => $gw_obj): ?>
                  <option value="<?php echo esc_attr($gw_id); ?>">
                    <?php echo esc_html($gw_obj->get_title() . " ($gw_id)"); ?>
                  </option>
                <?php endforeach; ?>
              </select>
            </td>
          </tr>
        </table>

        <p><button class="button button-primary">Add rule</button></p>
      </form>

      <hr>

      <h2>Saved rules</h2>
      <?php if (empty($rules)): ?>
        <p>No rules yet.</p>
      <?php else: ?>
        <form method="post">
          <?php wp_nonce_field(self::NONCE_ACTION, 'codarab_bgc_nonce'); ?>
          <input type="hidden" name="codarab_bgc_action" value="delete_rules">

          <table class="widefat striped">
            <thead><tr><th style="width:60px;">Del</th><th>Country</th><th>Gateway</th></tr></thead>
            <tbody>
            <?php foreach ($rules as $i => $r): ?>
              <tr>
                <td><input type="checkbox" name="delete[]" value="<?php echo esc_attr($i); ?>"></td>
                <td><?php echo esc_html($r['country']); ?></td>
                <td><?php echo esc_html($r['gateway']); ?></td>
              </tr>
            <?php endforeach; ?>
            </tbody>
          </table>

          <p style="margin-top:12px;"><button class="button">Delete selected</button></p>
        </form>
      <?php endif; ?>
    </div>
    <?php
  }

  public function handle_save() {
    if (!is_admin() || !current_user_can('manage_woocommerce')) return;
    if (empty($_POST['codarab_bgc_action'])) return;
    if (empty($_POST['codarab_bgc_nonce']) || !wp_verify_nonce($_POST['codarab_bgc_nonce'], self::NONCE_ACTION)) return;

    $rules = $this->get_rules();
    $action = sanitize_text_field($_POST['codarab_bgc_action']);

    if ($action === 'add_rule') {
      $country = isset($_POST['country']) ? strtoupper(sanitize_text_field($_POST['country'])) : '';
      $gateway = isset($_POST['gateway']) ? sanitize_text_field($_POST['gateway']) : '';
      if ($country && $gateway) {
        $rules[] = ['country' => $country, 'gateway' => $gateway];
        update_option(self::OPT, $rules, false);
      }
      wp_safe_redirect(add_query_arg(['page'=>'codarab-block-gateways-country','saved'=>1], admin_url('admin.php')));
      exit;
    }

    if ($action === 'delete_rules') {
      $delete = isset($_POST['delete']) && is_array($_POST['delete']) ? array_map('intval', $_POST['delete']) : [];
      foreach ($delete as $idx) if (isset($rules[$idx])) unset($rules[$idx]);
      $rules = array_values($rules);
      update_option(self::OPT, $rules, false);

      wp_safe_redirect(add_query_arg(['page'=>'codarab-block-gateways-country','saved'=>1], admin_url('admin.php')));
      exit;
    }
  }
}

new CODARAB_Block_Gateways_By_Country();
